//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQuick 2.2
import QtGraphicalEffects 1.0

import wesual.Controls 1.0
import wesual.Ui       1.0

PopupAnchorItem {
    id : menu

    //! pass if needed to have acces to the propeties from the calling element
    property Item           targetItem

    //! Actions are the list entries to be displayed
    property list<Action> items

    //! use mapToItem or mapFromItem to get the position in the menu parent
    //! coordinate system if used by delegates click handler
    //!
    //! menu.clickPosition = delegate.mapToItem(view, mouse.x, mouse.y);
    //! menu.clickPosition = view.mapFromItem(delegate, mouse.x, mouse.y);
    property var  clickPosition

    property bool sanitize     : false
    property int  contentWidth : 150

    //! indices vor separation line
    property var separationIndices : []

    function sanitizeTopLeftCornerPosition() {
        menu.popupVisible = false;

        if (menu.distanceBottom < menu.popupItem.height)
            menu.y = Math.ceil(clickPosition.y - menu.popupItem.height);

        if (menu.distanceRight < menu.popupItem.width)
            menu.x = Math.ceil(clickPosition.x - menu.popupItem.width);

        menu.sanitize = false;
        menu.popupVisible = true;
    }

    onDistanceRightChanged : {
        // update position only at reopen by mouse click
        if(!menu.sanitize)
            return;

        sanitizeTopLeftCornerPosition();
    }

    onPopupVisibleChanged : {
        if (popupItem && popupVisible)
            popupItem.focus = true;
        else {
            if (targetItem) {
                targetItem.forceActiveFocus();
            }
        }
    }

    onClickPositionChanged : {
        menu.y = Math.ceil(clickPosition.y);
        menu.x = Math.ceil(clickPosition.x);

        menu.sanitize = true;
    }

    popup : Rectangle {
        id : contentWrapper
        implicitWidth : Math.max(menu.contentWidth, content.width + 6)
        implicitHeight : content.height + 6

        border {
            color : UI.color(UI.PrimaryControlBorder)
            width : 1
        }
        opacity : 0

        Column {
            id : content

            width : Math.max(menu.contentWidth - 6, childrenRect.width)
            height : childrenRect.height

            anchors.centerIn : parent

            Repeater {
                id : repeater

                model : menu.items
                width : {
                    var max = 0;
                    for (var i = 0; i < count; ++i) {
                        max = Math.max(max, itemAt(i).implicitWidth);
                    }
                    return max;
                }

                delegate : Rectangle {
                    id : delegate

                    //visible : modelData.enabled
                    width : parent.width
                    implicitWidth : icon.width + textItem.implicitWidth + 21
                    height : 32

                    color : {
                        if (!modelData.enabled)
                            return UI.color(UI.PrimaryBase);

                        if (mouseArea.pressed)
                            return UI.color(UI.PrimaryPress);
                        else if (mouseArea.containsMouse)
                            return UI.color(UI.PrimaryHover)
                        else
                            return UI.color(UI.PrimaryBase)
                    }

                    Behavior on color {
                        ColorAnimation { duration: 0 }
                    }

                    UiIcon {
                        id : icon

                        anchors.verticalCenter : parent.verticalCenter
                        x : 7
                        iconSize : UiIcon.Icon_24x24
                        iconName : modelData.iconName
                        visible  : modelData.enabled
                    }
                    HueSaturation {
                        anchors.fill : icon
                        source       : icon
                        saturation   : -1.0
                        lightness    : 0.79
                        visible      : !modelData.enabled
                    }

                    Text {
                        id : textItem
                        anchors.left : icon.right
                        anchors.leftMargin : 7
                        height : parent.height
                        text : modelData.text
                        horizontalAlignment : Qt.AlignLeft
                        verticalAlignment : Qt.AlignVCenter

                        font {
                            family : UI.fontFamily(UI.PrimaryFont)
                            weight : UI.fontWeight(UI.PrimaryFont)
                            pixelSize : 14
                        }
                        elide : Qt.ElideRight
                        color : modelData.enabled
                                ? UI.color(UI.PrimaryTextColor)
                                : UI.color(UI.DisabledText)
                    }

                    MouseArea {
                        id : mouseArea
                        anchors.fill : delegate
                        hoverEnabled : true
                        acceptedButtons : Qt.LeftButton | Qt.RightButton
                        onClicked : {
                            if (modelData.enabled) {
                                menu.hide();
                                modelData.trigger();
                            }
                        }
                    }

                    Rectangle {
                        id : separationLine
                        width  : parent.width - 8
                        height : 1
                        color  : UI.color(UI.SecondarySeparationLine)
                        anchors.horizontalCenter : parent.horizontalCenter
                        anchors.bottom : parent.bottom
                        visible : {
                            if (separationIndices.length > 0 &&
                                separationIndices[0] === index) {
                                separationIndices.shift();
                                return true;
                            } else {
                                return false;
                            }

                        }

                    }
                } // delegate
            }
        }


        states : [
            State {
                name : "visible"
                when : menu.popupVisible

                PropertyChanges {
                    target : contentWrapper
                    opacity : 1
                }
            }
        ]

        transitions : Transition {
            NumberAnimation {
                property : "opacity"
                duration : 200
            }
        }
    }
}
